function spring_mass_walking
%% Tabula Rasa

clear all
close all
clc

%% Parameters

l0 = 1;		%uncompressed spring length [m]
m = 80;		%mass [kg]
k = 23000;		%spring stiffness [N/m]
g = 9.81;		%acceleration due to gravity [m/s^2]
alpha0 = 75;	%landing angle of leg [deg]

alpha0 = alpha0*pi/180;

%% Initial Conditions

x = 0;
y = 1;
x_dot = 1.3;
y_dot = 0;

% x1 = -l0*cos(alpha0);
x0 = 0;

%% State variable
traj = [];
traj_temp =[];
q = [x y x_dot y_dot]';

%% Transition Conditions

foot_impact = odeset('Events', @double_stance);
push_off = odeset('Events', @single_stance);
peak = odeset('Events', @apex);

%% Simulation
for i = 1:11
    [~,q] = ode45(@EoM_single, [0,5], q,foot_impact);
    traj_temp = [traj_temp; q];
    q = q(end,:);
    x1 = q(1) + l0*cos(alpha0);
    [~,q] = ode45(@EoM_double, [0,5], q,push_off);
    traj_temp = [traj_temp; q];
    traj_temp(:,1) = traj_temp(:,1) + x0;
    traj = [traj; traj_temp];
    x0 = x0 + x1;
    q = [traj(end,1) - x0 traj(end,2:4)]';
    [~,q] = ode45(@EoM_single, [0,5], q,peak);
    traj_temp = q;
    q = q(end,:);
end
traj_temp(:,1) = traj_temp(:,1) + x0;
traj = [traj; traj_temp];
plot(traj(:,1),traj(:,2))
title('Trajectory of 10 steps for spring mass walking')
xlabel('x (m)')
ylabel('y (m)')

%% Output Figure - uncomment to print a .eps file.
% sprin_mass_walking_traj.eps -depsc2

%% Equation of motion for single support phase
function output = EoM_single(~,q)
	
	omega = sqrt(k/m);
	output(1,1) = q(3);
	output(2,1) = q(4);
	output(3,1) = q(1)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1);
	output(4,1) = q(2)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1) - g;
end
%% Equation of motion for double support phase
function output = EoM_double(~,q)
    
    
    output(1,1) = q(3);
    output(2,1) = q(4);
    output(3,1) = -k*(q(1) - l0*q(1)*(q(1)^2 + q(2)^2)^-0.5 + (q(1) - x1) - l0*(q(1) - x1)*((q(1) - x1)^2 + q(2)^2)^-0.5)/m;
    output(4,1) = -k*(2*q(2) - l0*q(2)*(q(1)^2 + q(2)^2)^-0.5 - l0*q(2)*((q(1) - x1)^2 + q(2)^2)^-0.5)/m - g;
end
%% End of Step conditions
function [impact,terminate,direction] = apex(~,q)
    impact = q(4);
	terminate = 1;
	direction = -1;
end
%% Push-off Conditions
function [impact,terminate,direction] = single_stance(~,q)
    impact = sqrt((q(1)^2 + q(2)^2)) - l0;
	terminate = 1;
	direction = 1;
end
%% Touchdown Conditions
function [impact,terminate,direction] = double_stance(~,q)

    impact = q(2) - l0*sin(alpha0);
	terminate = 1;
	direction = -1;
end
end